import { Switch, Route, useLocation } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { ThemeProvider } from "@/components/ThemeProvider";
import { SidebarProvider, SidebarTrigger } from "@/components/ui/sidebar";
import { AppSidebar } from "@/components/AppSidebar";
import { useAuth } from "@/hooks/use-auth";
import { useEffect } from "react";
import NotFound from "@/pages/not-found";
import Landing from "@/pages/Landing";
import Login from "@/pages/Login";
import Dashboard from "@/pages/Dashboard";
import PlanDetail from "@/pages/PlanDetail";
import AIAssistant from "@/pages/AIAssistant";
import Notifications from "@/pages/Notifications";
import Profile from "@/pages/Profile";
import Settings from "@/pages/Settings";
import Admin from "@/pages/Admin";
import About from "@/pages/About";
import Pricing from "@/pages/Pricing";
import SavedHistory from "@/pages/SavedHistory";
import MarketingHeader from "@/components/MarketingHeader";
import MarketingFooter from "@/components/MarketingFooter";

function Router() {
  const { isAuthenticated, isLoading } = useAuth();
  const [location, setLocation] = useLocation();

  // Show loading only for a brief moment, don't block the entire app
  if (isLoading) {
    return (
      <div className="min-h-screen bg-background">
        <MarketingHeader />
        <div className="flex items-center justify-center h-[calc(100vh-4rem)] pt-16">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary mx-auto mb-4"></div>
            <p className="text-muted-foreground">Loading...</p>
          </div>
        </div>
      </div>
    );
  }

  if (!isAuthenticated) {
    return (
      <div className="min-h-screen bg-background">
        <MarketingHeader />
        <Switch>
          <Route path="/" component={Landing} />
          <Route path="/login" component={Login} />
          <Route path="/about" component={About} />
          <Route path="/pricing" component={Pricing} />
          <Route component={Landing} />
        </Switch>
        <MarketingFooter />
      </div>
    );
  }

  return (
    <SidebarProvider>
      <div className="flex min-h-screen w-full flex-col">
        <MarketingHeader />
        <div className="flex flex-1 pt-16">
          <AppSidebar />
          <div className="flex flex-1 flex-col">
            <main className="flex-1 overflow-auto">
              <Switch>
                <Route path="/" component={Landing} />
                <Route path="/about" component={About} />
                <Route path="/pricing" component={Pricing} />
                <Route path="/dashboard" component={Dashboard} />
                <Route path="/plans" component={Dashboard} />
                <Route path="/plan/:id" component={PlanDetail} />
                <Route path="/ai-assistant" component={AIAssistant} />
                <Route path="/saved-history" component={SavedHistory} />
                <Route path="/notifications" component={Notifications} />
                <Route path="/profile" component={Profile} />
                <Route path="/settings" component={Settings} />
                <Route path="/admin" component={Admin} />
                <Route component={NotFound} />
              </Switch>
            </main>
            <MarketingFooter />
          </div>
        </div>
      </div>
    </SidebarProvider>
  );
}

function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <ThemeProvider defaultTheme="light">
        <TooltipProvider>
          <Toaster />
          <Router />
        </TooltipProvider>
      </ThemeProvider>
    </QueryClientProvider>
  );
}

export default App;
