import { Card } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { Button } from "@/components/ui/button";
import { MoreVertical, Eye, Edit, Trash2 } from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

interface PlanCardProps {
  id: string;
  title: string;
  description?: string;
  progress: number;
  stepCount: number;
  completedSteps: number;
  lastUpdated: string;
  onView?: () => void;
  onEdit?: () => void;
  onDelete?: () => void;
}

export function PlanCard({
  id,
  title,
  description,
  progress,
  stepCount,
  completedSteps,
  lastUpdated,
  onView,
  onEdit,
  onDelete,
}: PlanCardProps) {
  return (
    <Card className="p-6 hover-elevate transition-all duration-150 cursor-pointer" data-testid={`card-plan-${id}`}>
      <div className="space-y-4">
        <div className="flex items-start justify-between gap-4">
          <div className="flex-1 min-w-0">
            <h3 className="text-lg font-semibold truncate" data-testid={`text-plan-title-${id}`}>
              {title}
            </h3>
            {description && (
              <p className="text-sm text-muted-foreground line-clamp-2 mt-1">
                {description}
              </p>
            )}
          </div>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" size="icon" data-testid={`button-plan-menu-${id}`}>
                <MoreVertical className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem onClick={onView} data-testid={`button-view-${id}`}>
                <Eye className="h-4 w-4 mr-2" />
                View
              </DropdownMenuItem>
              <DropdownMenuItem onClick={onEdit} data-testid={`button-edit-${id}`}>
                <Edit className="h-4 w-4 mr-2" />
                Edit
              </DropdownMenuItem>
              <DropdownMenuItem onClick={onDelete} className="text-destructive" data-testid={`button-delete-${id}`}>
                <Trash2 className="h-4 w-4 mr-2" />
                Delete
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>

        <div className="space-y-2">
          <div className="flex items-center justify-between text-sm">
            <span className="text-muted-foreground">Progress</span>
            <span className="font-semibold" data-testid={`text-progress-${id}`}>{progress}%</span>
          </div>
          <Progress value={progress} className="h-2" />
        </div>

        <div className="flex items-center justify-between text-sm text-muted-foreground">
          <span data-testid={`text-steps-${id}`}>
            {completedSteps} of {stepCount} steps
          </span>
          <span className="text-xs" data-testid={`text-updated-${id}`}>{lastUpdated}</span>
        </div>
      </div>
    </Card>
  );
}
